<?php

namespace App\Http\Controllers;

use App\Models\Organization;
use App\Models\OrganizationInvitation;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Inertia\Inertia;
use Carbon\Carbon;
use App\Http\Requests\InvitationRequest;
use Illuminate\Support\Facades\Mail;
use App\Mail\OrganizationInvitationMail as InvitationMail;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class OrganizationInvitationController extends Controller
{
    use AuthorizesRequests;

    public function index(Organization $organization)
    {
        $this->authorize('manageMembers', $organization);

        return Inertia::render('Organizations/Invitations/Index', [
            'organization' => $organization->load('owner'),
            'invitations' => $organization->invitations()
                ->with('inviter')
                ->get()
                ->map(function ($invitation) {
                    return array_merge($invitation->toArray(), [
                        'status' => $invitation->getStatus(),
                        'expires_at_formatted' => $invitation->expires_at->format('M d, Y H:i'),
                    ]);
                }),
            'canManageMembers' => auth()->user()->can('manageMembers', $organization),
        ]);
    }

    public function store(InvitationRequest $request, Organization $organization)
    {
        $this->authorize('manageMembers', $organization);

        // Check if user is already a member
        if ($organization->users()->where('email', $request->email)->exists()) {
            return back()->withErrors([
                'email' => 'This user is already a member of the organization.',
            ]);
        }

        // Check for pending invitation
        if ($organization->invitations()->where('email', $request->email)->whereNull('accepted_at')->whereNull('declined_at')->exists()) {
            return back()->withErrors([
                'email' => 'An invitation has already been sent to this email address.',
            ]);
        }

        $invitation = $organization->invitations()->create([
            'email' => $request->email,
            'role' => $request->role,
            'invited_by' => auth()->id(),
            'token' => Str::random(32),
            'expires_at' => Carbon::parse($request->expires_at),
        ]);

        // Send invitation email
        Mail::to($request->email)->send(new InvitationMail($invitation));

        return back()->with('success', 'Invitation sent successfully.');
    }

    public function resend(Organization $organization, OrganizationInvitation $invitation)
    {
        $this->authorize('manageMembers', $organization);

        if (!$invitation->isPending()) {
            return back()->withErrors([
                'email' => 'This invitation cannot be resent.',
            ]);
        }

        // Update expiration and token
        $invitation->update([
            'token' => Str::random(32),
            'expires_at' => now()->addDays(7),
        ]);

        // Resend invitation email
        Mail::to($invitation->email)->send(new InvitationMail($invitation));

        return back()->with('success', 'Invitation resent successfully.');
    }

    public function destroy(Organization $organization, OrganizationInvitation $invitation)
    {
        $this->authorize('manageMembers', $organization);

        if (!$invitation->isPending()) {
            return back()->withErrors([
                'email' => 'This invitation cannot be cancelled.',
            ]);
        }

        $invitation->delete();

        return back()->with('success', 'Invitation cancelled successfully.');
    }

    public function accept(string $token)
    {
        $invitation = OrganizationInvitation::where('token', $token)
            ->whereNull('accepted_at')
            ->whereNull('declined_at')
            ->firstOrFail();

        if ($invitation->isExpired()) {
            return redirect()->route('dashboard')
                ->with('error', 'This invitation has expired.');
        }

        // Add user to organization
        $invitation->organization->users()->attach(auth()->id(), [
            'role' => $invitation->role,
            'joined_at' => now(),
        ]);

        $invitation->update(['accepted_at' => now()]);

        return redirect()->route('organizations.show', $invitation->organization)
            ->with('success', 'You have joined the organization successfully.');
    }

    public function decline(string $token)
    {
        $invitation = OrganizationInvitation::where('token', $token)
            ->whereNull('accepted_at')
            ->whereNull('declined_at')
            ->firstOrFail();

        $invitation->update(['declined_at' => now()]);

        return redirect()->route('dashboard')
            ->with('success', 'Invitation declined successfully.');
    }
} 